<?php
// --- MOB DROP GROUP FUNCTIONS ---
function parse_mob_drop_file($filename) {
    $groups = [];
    $current_group = null;
    $current_drops = [];
    if (!file_exists($filename)) return [];
    $lines = file($filename);
    foreach ($lines as $line) {
        $line = trim($line);
        if (empty($line)) continue;
        
        // --- FIX 1: Make Group name parsing more robust ---
        if (stripos($line, 'Group') === 0) { // Use stripos for case-insensitive check
            if ($current_group) {
                $groups[] = $current_group;
            }
            // Extract name after "Group" or "group"
            $parts = preg_split('/\s+/', $line, 2);
            $current_group = [
                'name' => isset($parts[1]) ? $parts[1] : '',
                'mob' => '',
                'type' => 'drop', // Default type
                'drops' => [],
            ];
        } elseif ($line == '{') {
            $current_drops = [];
        } elseif ($line == '}') {
            if ($current_group) {
                $current_group['drops'] = $current_drops;
            }
        } 
        
        // --- FIX 2: Make Mob ID parsing case-insensitive ---
        elseif (stripos($line, 'Mob') === 0) { // Use stripos for case-insensitive check
            $parts = preg_split('/\s+/', $line, 2);
            $current_group['mob'] = isset($parts[1]) ? trim($parts[1]) : '';
        }

        // --- FIX 3: Add logic to parse 'type' and 'kill_drop' ---
        elseif (stripos($line, 'type') === 0) {
            $parts = preg_split('/\s+/', $line, 2);
            $current_group['type'] = isset($parts[1]) ? trim($parts[1]) : 'drop';
        }
        elseif (stripos($line, 'kill_drop') === 0) {
             // We don't need to store it, but we acknowledge it so it doesn't cause issues
        }

        // --- FIX 4: Update regex to accept 4 OR 5 columns ---
        // This regex now matches lines with 4 or 5 columns of numbers
        elseif (preg_match('/^\d+\s+\d+\s+\d+\s+[\d\.]+(\s+\d+)?/', $line)) {
            $drop_data = preg_split('/\s+/', $line);
            // If there are 5 columns, we only take the first 4 to match the editor's structure
            if (count($drop_data) > 4) {
                $drop_data = array_slice($drop_data, 0, 4);
            }
            $current_drops[] = $drop_data;
        }
    }
    if ($current_group) {
        $groups[] = $current_group;
    }
    return $groups;
}

function getItemIconById($itemId, $itemListFile = 'item_list.txt') {
    $itemListFile = trim($itemListFile);
    if (!file_exists($itemListFile)) return '';
    $lines = file($itemListFile);
    foreach ($lines as $line) {
        $line = trim($line);
        if ($line === '') continue;
        $parts = preg_split('/\s+/', $line);
        if (count($parts) >= 2 && $parts[0] == $itemId) {
            return $parts[1];
        }
    }
    return '';
}

// NEW: Get item name from names.txt
function getItemNameById($itemId, $namesFile = 'names.txt') {
    $namesFile = trim($namesFile);
    if (!file_exists($namesFile)) return '';
    $lines = file($namesFile);
    foreach ($lines as $line) {
        $line = trim($line);
        if ($line === '' || strpos($line, 'VNUM') === 0) continue;
        $parts = preg_split('/\s+/', $line, 2);
        if (count($parts) >= 2 && $parts[0] == $itemId) {
            return $parts[1];
        }
    }
    return '';
}

// Mob image path function
function getMobImagePath($mobId) {
    $mobId = trim($mobId);
    $imgPath = "icon/mob/$mobId.png";
    $defaultPath = "icon/mob/no_img.png";
    return file_exists(__DIR__ . "/$imgPath") ? $imgPath : $defaultPath;
}

function update_mob_drop_file($filename, $groups) {
    $lines = [];
    foreach ($groups as $group) {
        $lines[] = "Group " . $group['name'];
        $lines[] = "{";
        if ($group['mob']) $lines[] = "Mob " . $group['mob'];
        $lines[] = "Type\tdrop";
        foreach ($group['drops'] as $drop) {
            $lines[] = implode("\t", $drop);
        }
        $lines[] = "}";
    }
    $result = file_put_contents($filename, implode(PHP_EOL, $lines) . PHP_EOL);
    return $result !== false;
}

// function trigger_go_event($group_idx) {
//     $url = "http://localhost:8080/event";
//     $data = json_encode(["group_idx" => $group_idx, "timestamp" => time()]);
//     $opts = [
//         "http" => [
//             "method" => "POST",
//             "header" => "Content-Type: application/json\r\n",
//             "content" => $data
//         ]
//     ];
//     $context = stream_context_create($opts);
//     @file_get_contents($url, false, $context);
// }

$filename = __DIR__ . "/mob_drop_item.txt";
$itemListPath = __DIR__ . "/item_list.txt";
$namesPath = __DIR__ . "/names.txt";
if (!file_exists($filename)) {
    die("<div class='fatal-error'><b>mob_drop_item.txt file not found!</b><br>Please add the file to the panel folder.</div>");
}
$groups = parse_mob_drop_file($filename);
if (!isset($groups) || !is_array($groups)) $groups = [];

$selected = isset($_GET['group']) ? intval($_GET['group']) : null;

// Add Mob Group
if ($_SERVER['REQUEST_METHOD'] === 'POST' && isset($_POST['add_group'])) {
    $group_name = trim($_POST['group_name']);
    $mob_id = trim($_POST['mob_id']);
    if ($group_name !== '' && $mob_id !== '') {
        $groups[] = [
            'name' => $group_name,
            'mob' => $mob_id,
            'drops' => []
        ];
        update_mob_drop_file($filename, $groups);
        header("Location: index.php?group=" . (count($groups) - 1) . "&success=group");
        exit;
    }
}

// Add new drop
if ($_SERVER['REQUEST_METHOD'] === 'POST' && isset($_POST['add_drop'])) {
    $group_idx = intval($_POST['group']);
    $drop_num = count($groups[$group_idx]['drops']) + 1;
    $new_drop = [
        $drop_num,
        $_POST['item_id'],
        $_POST['amount'],
        $_POST['rate']
    ];
    $groups[$group_idx]['drops'][] = $new_drop;
    update_mob_drop_file($filename, $groups);
    header("Location: index.php?group=$group_idx&success=add");
    exit;
}

// Edit drop
if ($_SERVER['REQUEST_METHOD'] === 'POST' && isset($_POST['edit_drop'])) {
    $group_idx = intval($_POST['group']);
    $drop_idx = intval($_POST['drop_idx']);
    $groups[$group_idx]['drops'][$drop_idx] = [
        $drop_idx + 1,
        $_POST['item_id'],
        $_POST['amount'],
        $_POST['rate']
    ];
    update_mob_drop_file($filename, $groups);
    header("Location: index.php?group=$group_idx&success=edit");
    exit;
}

// Delete drop (automatically update ordering)
if (isset($_GET['delete_drop']) && $selected !== null) {
    $drop_idx = intval($_GET['delete_drop']);
    array_splice($groups[$selected]['drops'], $drop_idx, 1);
    foreach ($groups[$selected]['drops'] as $i => &$drop) {
        $drop[0] = $i + 1;
    }
    update_mob_drop_file($filename, $groups);
    header("Location: index.php?group=$selected&success=delete");
    exit;
}

// BULK EDIT - save
if ($_SERVER['REQUEST_METHOD'] === 'POST' && isset($_POST['bulk_edit_save'])) {
    $group_idx = intval($_POST['group']);
    $new_drops = [];
    foreach ($_POST['bulk_drop_num'] as $i => $num) {
        $item = $_POST['bulk_item_id'][$i];
        $amount = $_POST['bulk_amount'][$i];
        $rate = $_POST['bulk_rate'][$i];
        if (trim($num)==='' && trim($item)==='' && trim($amount)==='' && trim($rate)==='') continue;
        $new_drops[] = [
            '', $item, $amount, $rate
        ];
    }
    foreach ($new_drops as $i => &$drop) {
        $drop[0] = $i + 1;
    }
    $groups[$group_idx]['drops'] = $new_drops;
    if (!update_mob_drop_file($filename, $groups)) {
        die("File could not be written! Please check the file permissions.");
    }
    // trigger_go_event($group_idx); // Enable to notify Go service
    header("Location: index.php?group=$group_idx&success=bulk");
    exit;
}

// --- Preparing group data for JS ---
function turkishLower($str) {
    $letters = ["İ"=>"i","I"=>"ı","Ş"=>"ş","Ğ"=>"ğ","Ü"=>"ü","Ö"=>"ö","Ç"=>"ç"];
    $str = str_replace(array_keys($letters), array_values($letters), $str);
    return mb_strtolower($str, 'UTF-8');
}
$groupData = [];
foreach ($groups as $idx => $group) {
    $groupData[] = [
        "idx" => $idx,
        "name" => $group['name'],
        "mob" => $group['mob'],
        "items" => "",
        "selected" => ($selected === $idx)
    ];
}
?>
<!DOCTYPE html>
<html lang="en">
<head>
    <meta charset="UTF-8">
    <title>Mob Drop Admin Panel</title>
    <meta name="viewport" content="width=device-width,initial-scale=1">
    <link rel="stylesheet" href="style.css?v=12">
    <style>
        body { background: linear-gradient(120deg,#e8f3ff 60%,#f8fafc 100%); }
        .admin-header-logo { box-shadow: 0 4px 12px #2581c633; }
        .dashboard-wrapper { box-shadow: 0 4px 32px #2581c614; border-radius: 20px; }
        .admin-content { box-shadow: 0 0 40px #2581c613,0 2px 22px #324a5e14;}
        .group-list-item { font-size: 1.07rem; }
        .group-list-link { transition: box-shadow .2s;}
        .group-list-item.active-group { border:2px solid #43d47b;}
        .btn-edit,.btn-delete,.btn-save,.btn-cancel,.btn-add { font-size: 1.11rem; border-radius:12px;}
        .drop-table { box-shadow: 0 4px 32px #43d47b22;}
        .group-search { letter-spacing: 0.5px; }
        .admin-content-title { font-size: 1.65rem; }
        .tooltip-wrap {
            position: relative;
            display: inline-block;
        }
        .tooltip {
            visibility: hidden;
            background: #324a5e;
            color: #fff;
            text-align: center;
            border-radius: 8px;
            padding: 8px 14px;
            position: absolute;
            z-index: 222;
            bottom: 120%;
            left: 50%;
            transform: translateX(-50%);
            font-size: 1.01rem;
            min-width: 90px;
            box-shadow: 0 4px 16px #324a5e44;
            opacity: 0;
            pointer-events: none;
            transition: opacity 0.18s;
            white-space: pre-line;
        }
        .tooltip-wrap:hover .tooltip {
            visibility: visible;
            opacity: 1;
        }
        @media (max-width: 1000px) {
            .dashboard-wrapper { flex-direction:column; }
            .admin-content { margin:14px 2px; padding:14px 2vw; }
        }
        @media (max-width:600px) {
            .admin-sidebar { padding:8px 0 5px 0; }
        }
    </style>
</head>
<body>
    <audio id="successSound" src="new-notification.mp3" preload="auto"></audio>
<header class="admin-header">
    <div class="admin-header-left">
        <img src="https://gf1.geo.gfsrv.net/cdn98/191b803adbf82f4b8febe3a2c38c2c.ico" alt="Drop" class="admin-header-logo">
        <span class="admin-header-title">Mob Drop Admin Panel</span>
        <a href="mob_proto_edit.php" class="admin-header-btn" style="margin-left:20px; padding:6px 14px; background:#3b7ddd; color:white; border-radius:4px; text-decoration:none; font-weight:500;">🗎 Edit Mob Proto</a>
        <a href="special_item_group_edit.php" class="admin-header-btn" style="margin-left:20px; padding:6px 14px; background:#3b7ddd; color:white; border-radius:4px; text-decoration:none; font-weight:500;">🗎 Edit Chest Items</a>
    </div>
    <div class="admin-header-right">
        <span class="admin-header-user">👨‍💻 Administrator</span>
        <span class="admin-header-role">Panel</span>
    </div>
</header>
    <div id="toastArea"></div>
    <div class="dashboard-wrapper">
<aside class="admin-sidebar">
    <h2>Mob Groups</h2>
    <form method="post" class="add-group-form" style="margin-bottom:14px;">
        <input type="text" name="group_name" placeholder="New group name" required class="group-input" autocomplete="off">
        <input type="text" name="mob_id" placeholder="Mob ID" required class="group-input" autocomplete="off">
        <button type="submit" name="add_group" class="btn-add-group">+ Add New Group</button>
    </form>
    <form method="get" style="margin-bottom: 14px;">
        <input type="text" id="groupSearch" name="search" class="group-search" 
               placeholder="Search by group name or Mob ID..." autocomplete="off"
               value="<?= isset($_GET['search']) ? htmlspecialchars($_GET['search']) : '' ?>">
        <?php if ($selected !== null): ?>
            <input type="hidden" name="group" value="<?= $selected ?>">
        <?php endif; ?>
    </form>
<ul class="group-list-ul" id="groupList">
    <?php
    $searchQuery = isset($_GET['search']) ? mb_strtolower(trim($_GET['search'])) : '';
    $groupCount = 0;
    foreach ($groups as $idx => $group):
        if ($searchQuery &&
            strpos(mb_strtolower($group['name']), $searchQuery) === false &&
            strpos(mb_strtolower($group['mob']), $searchQuery) === false) continue;
        $groupCount++;
        $mobImgPath = getMobImagePath($group['mob']);
    ?>
<li class="group-list-item<?= ($selected === $idx) ? ' active-group' : '' ?>"
    data-group-idx="<?= $idx ?>"
    data-group-name="<?= htmlspecialchars($group['name']) ?>"
    data-mob-id="<?= htmlspecialchars($group['mob']) ?>"
    onclick="window.location.href='?group=<?= $idx ?>';"
    style="cursor:pointer;display:flex;align-items:center;">
    <img src="<?= htmlspecialchars($mobImgPath) ?>" alt="Mob" style="width:34px;height:34px;margin-right:8px;border-radius:8px;box-shadow:0 2px 8px #2581c633;">
    <span class="group-name"><?= htmlspecialchars($group['name']) ?></span>
    <span class="group-id" style="margin-left:7px;">(Mob ID: <?= htmlspecialchars($group['mob']) ?>)</span>
</li>
    <?php endforeach; ?>
    <?php if ($groupCount == 0): ?>
        <li><em>No groups found.</em></li>
    <?php endif; ?>
</ul>
</aside>
<script>
document.addEventListener("DOMContentLoaded", function() {
    var groupSearch = document.getElementById("groupSearch");
    var groupList = document.getElementById("groupList");
    if (groupSearch && groupList) {
        groupSearch.addEventListener("input", function(){
            var query = groupSearch.value.trim().toLowerCase();
            groupList.querySelectorAll(".group-list-item").forEach(function(li){
                var name = (li.getAttribute('data-group-name') || "").toLowerCase();
                var mobid = (li.getAttribute('data-mob-id') || "").toLowerCase();
                if(query === "" || name.includes(query) || mobid.includes(query)) {
                    li.style.display = "";
                }else{
                    li.style.display = "none";
                }
            });
        });
    }
    var activeGroup = groupList.querySelector(".active-group");
    if (activeGroup) {
        setTimeout(function(){
            activeGroup.scrollIntoView({block: "center", behavior: "smooth"});
        }, 150);
    }
});
</script>
        <main class="admin-content">
            <?php
            if (isset($_GET['success'])) {
                $msg = "Record ";
                switch ($_GET['success']) {
                    case "group": $msg .= "added successfully"; break;
                    case "add": $msg .= "added successfully"; break;
                    case "edit": $msg .= "updated successfully"; break;
                    case "delete": $msg = "Record deleted"; break;
                    case "bulk": $msg = "Bulk records saved successfully"; break;
                    default: $msg = "Success!"; break;
                }
                echo '<div class="toast-notify toast-success" id="toastNotify">'
                    . '<span class="toast-icon">&#10003;</span>'
                    . '<span>' . htmlspecialchars($msg) . '</span>'
                    . '<span class="toast-close" title="Close" onclick="this.parentElement.remove();">&times;</span>'
                    . '</div>';
            }
            ?>
            <?php if ($selected !== null && isset($groups[$selected])): ?>
            <div class="mob-image-area" style="text-align:center; margin-bottom:24px;">
                <?php
                $mobId = $groups[$selected]['mob'];
                $mobImgPath = getMobImagePath($mobId);
                ?>
                <img src="<?= htmlspecialchars($mobImgPath) ?>" alt="Mob Image" style="width:220px;max-width:100%;border-radius:16px;box-shadow:0 2px 16px #2581c633;">
                <div style="margin-top:7px; color:#2581c6; font-size:1.11rem;">
                    Mob ID: <b><?= htmlspecialchars($mobId) ?></b>
                </div>
            </div>
                <h2 class="admin-content-title">
                    <?= htmlspecialchars($groups[$selected]['name']) ?> Drop List
                    <span style="font-size:1.05rem;color:#43d47b;font-weight:400;margin-left:10px;">(Mob ID: <?= htmlspecialchars($groups[$selected]['mob']) ?>)</span>
                </h2>
                <div style="margin-bottom:24px;">
                    <a href="?group=<?= $selected ?>&bulk_edit=1" class="btn-add" style="background:#e8f3ff;color:#2581c6;">Bulk Edit</a>
                </div>
                <?php if (isset($_GET['bulk_edit']) && $_GET['bulk_edit']==1): ?>
    <form method="post" class="bulk-edit-form">
        <input type="hidden" name="group" value="<?= $selected ?>">
        <div style="margin-bottom:10px;display:flex;gap:10px;align-items:center;">
            <label for="bulkRateChange" style="font-weight:600;color:#2581c6;">Bulk change drop rates:</label>
            <input type="number" id="bulkRateChange" value="50" min="1" max="100" style="width:70px;">
            <button type="button" onclick="changeBulkRates('decrease')" class="btn-delete">-%</button>
            <button type="button" onclick="changeBulkRates('increase')" class="btn-add">+%</button>
            <span style="font-size:0.95em;color:#888;">Ex: write 50 and click -% to reduce rates by 50%.</span>
        </div>
        <table class="drop-table">
            <thead>
                <tr>
                    <th>#</th>
                    <th>Item ID</th>
                    <th>Icon</th>
                    <th>Amount</th>
                    <th>Rate</th>
                </tr>
            </thead>
            <tbody>
            <?php
            $maxRows = max( count($groups[$selected]['drops'])+2, 5 );
            for ($i=0; $i < $maxRows; $i++):
                $drop = isset($groups[$selected]['drops'][$i]) ? $groups[$selected]['drops'][$i] : ["","","",""];
            ?>
                <tr>
                    <td><input type="text" name="bulk_drop_num[]" value="<?= $i+1 ?>" size="2" readonly></td>
                    <td><input type="text" name="bulk_item_id[]" value="<?= htmlspecialchars($drop[1]) ?>" size="5"></td>
                    <td>
                        <?php
                        $iconPath = getItemIconById($drop[1], $itemListPath);
                        $itemName = getItemNameById($drop[1], $namesPath);
                        if ($iconPath && file_exists(__DIR__ . '/' . $iconPath)) {
                            echo '<span class="tooltip-wrap">
                                    <img src="' . htmlspecialchars($iconPath) . '" alt="" class="icon-img">
                                    <span class="tooltip">' . htmlspecialchars($itemName) . '</span>
                                  </span>';
                        } elseif(trim($drop[1])!=="") {
                            echo '<span class="icon-missing tooltip-wrap">None<span class="tooltip">' . htmlspecialchars($itemName) . '</span></span>';
                        }
                        ?>
                    </td>
                    <td><input type="text" name="bulk_amount[]" value="<?= htmlspecialchars($drop[2]) ?>" size="3"></td>
                    <td><input type="text" name="bulk_rate[]" value="<?= htmlspecialchars($drop[3]) ?>" size="4" class="bulk-rate-input"></td>
                </tr>
            <?php endfor; ?>
            </tbody>
        </table>
        <input type="submit" name="bulk_edit_save" value="Save Bulk" class="btn-save">
        <a href="index.php?group=<?= $selected ?>" class="btn-cancel">Cancel</a>
    </form>
    <script>
    function changeBulkRates(mode) {
        var percent = parseFloat(document.getElementById('bulkRateChange').value) || 0;
        if (percent <= 0) return;
        var factor = (mode === 'decrease') 
            ? (1 - percent/100) 
            : (1 + percent/100);
        document.querySelectorAll('.bulk-rate-input').forEach(function(input){
            var val = parseFloat(input.value.replace(",", "."));
            if (isNaN(val)) return;
            var newVal = Math.round(val * factor * 10000) / 10000;
            input.value = newVal > 0 ? newVal : 0;
        });
    }
    </script>
                <?php else: ?>
                    <table class="drop-table" id="dropTable">
                        <thead>
                            <tr>
                                <th>#</th>
                                <th>Item ID</th>
                                <th>Icon</th>
                                <th>Amount</th>
                                <th>Rate</th>
                                <th>Action</th>
                            </tr>
                        </thead>
                        <tbody>
                        <?php foreach ($groups[$selected]['drops'] as $i => $drop): ?>
                        <tr>
                            <?php if (isset($_GET['edit_drop']) && $_GET['edit_drop'] == $i): ?>
                                <form method="post" class="edit-drop-form">
                                    <td><input type="text" name="drop_num" value="<?= $i+1 ?>" size="2" required readonly></td>
                                    <td><input type="text" name="item_id" value="<?= htmlspecialchars($drop[1]) ?>" size="5" required></td>
                                    <td>
                                        <?php
                                        $iconPath = getItemIconById($drop[1], $itemListPath);
                                        $itemName = getItemNameById($drop[1], $namesPath);
                                        if ($iconPath && file_exists(__DIR__ . '/' . $iconPath)) {
                                            echo '<span class="tooltip-wrap">
                                                    <img src="' . htmlspecialchars($iconPath) . '" alt="" class="icon-img">
                                                    <span class="tooltip">' . htmlspecialchars($itemName) . '</span>
                                                  </span>';
                                        } else {
                                            echo '<span class="icon-missing tooltip-wrap">None<span class="tooltip">' . htmlspecialchars($itemName) . '</span></span>';
                                        }
                                        ?>
                                    </td>
                                    <td><input type="text" name="amount" value="<?= htmlspecialchars($drop[2]) ?>" size="3" required></td>
                                    <td><input type="text" name="rate" value="<?= htmlspecialchars($drop[3]) ?>" size="4" required></td>
                                    <td class="table-action">
                                        <input type="hidden" name="group" value="<?= $selected ?>">
                                        <input type="hidden" name="drop_idx" value="<?= $i ?>">
                                        <button type="submit" name="edit_drop" class="btn-save">Save</button>
                                        <a href="index.php?group=<?= $selected ?>" class="btn-cancel">Cancel</a>
                                    </td>
                                </form>
                            <?php else: ?>
                                <td><?= $i+1 ?></td>
                                <td><?= htmlspecialchars($drop[1]) ?></td>
                                <td>
                                    <?php
                                    $iconPath = getItemIconById($drop[1], $itemListPath);
                                    $itemName = getItemNameById($drop[1], $namesPath);
                                    if ($iconPath && file_exists(__DIR__ . '/' . $iconPath)) {
                                        echo '<span class="tooltip-wrap">
                                                <img src="' . htmlspecialchars($iconPath) . '" alt="" class="icon-img">
                                                <span class="tooltip">' . htmlspecialchars($itemName) . '</span>
                                              </span>';
                                    } else {
                                        echo '<span class="icon-missing tooltip-wrap">None<span class="tooltip">' . htmlspecialchars($itemName) . '</span></span>';
                                    }
                                    ?>
                                </td>
                                <td><?= htmlspecialchars($drop[2]) ?></td>
                                <td><?= htmlspecialchars($drop[3]) ?></td>
                                <td class="table-action">
                                    <a href="index.php?group=<?= $selected ?>&edit_drop=<?= $i ?>" class="btn-edit">Edit</a>
                                    <a href="index.php?group=<?= $selected ?>&delete_drop=<?= $i ?>" class="btn-delete" onclick="return confirm('Are you sure you want to delete?')">Delete</a>
                                </td>
                            <?php endif; ?>
                        </tr>
                        <?php endforeach; ?>
                        </tbody>
                    </table>
                    <form method="post" class="add-form" id="addDropForm" style="margin-top:30px;">
                        <input type="hidden" name="group" value="<?= $selected ?>">
                        <input type="text" name="drop_num" value="<?= count($groups[$selected]['drops'])+1 ?>" placeholder="#" size="2" readonly disabled>
                        <input type="text" name="item_id" placeholder="Item ID" size="5" required>
                        <input type="text" name="amount" placeholder="Amount" size="3" required>
                        <input type="text" name="rate" placeholder="Rate" size="4" required>
                        <button type="submit" name="add_drop" class="btn-add">Add New Drop</button>
                    </form>
                <?php endif; ?>
            <?php else: ?>
                <div class="empty">
                    <img src="https://cdn-icons-png.flaticon.com/512/565/565547.png" width="96" alt="Select group">
                    <div>Please select a group or click the \"Edit\" button on the left.</div>
                </div>
            <?php endif; ?>
        </main>
    </div>
    <footer style="text-align:center;padding:18px 0;color:#2581c6;font-size:1rem;opacity:.8;">
        &copy; <?= date('Y') ?> Mob Drop Panel - Design & Code: <b>DepDarkMor</b><br><a>v1.0.13</a>
    </footer>
    <script>
    window.onload = function() {
        var toast = document.getElementById("toastNotify");
        var snd = document.getElementById("successSound");
        if (toast && snd) {
            try { snd.currentTime = 0; snd.play(); } catch(e) {}
            setTimeout(function(){ if (toast) toast.remove(); }, 3500);
        }
    };
    </script>
</body>
</html>
